#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 定义硬盘类

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2023/07/24 18:00 created

"""

from lib.core.raid_cards.raid_card import CardFactoryMap
from lib.core.tools.mcscli import McsCli
from lib.utils.log import SingleLog


class DriveFactoryMap(object):
    def __init__(self):
        self.drive_map = {
            "ata": AtaDrive,
            "scsi": ScsiDrive,
            "nvme": NvmeDrive
        }

    def get(self, drive_info):
        mn = drive_info.get("model_number")
        if mn in ("HWE62ST3960L005N", "HWE62ST31T9L005N"):
            drive_chunk_size = (65536, 4096)
        else:
            drive_chunk_size = (4096, 8192, 16384, 32768, 65536)
        drive_info["chunk_size"] = drive_chunk_size
        return self.drive_map.get(drive_info.get("protocol"))(drive_info)


class Drive(object):
    """
    Drive Entity Class
    """

    def __init__(self, info):
        self.info = info
        self.mcs = McsCli()
        self.log = SingleLog()
        self.card_info = CardFactoryMap().get(self.info.get("target", ""))

    def smart(self):
        pass

    def internal(self):
        """
        Returns:
            log_path:内部日志存放的路径，路径由mcs确定,若失败则返回空字符串
        """
        if self.card_info:
            card_chunk_size = self.card_info.chunk_size()
            available_chunk_size = set(self.chunk_size()) & set(card_chunk_size)
            if not available_chunk_size:
                self.log.error("collect internal log: card chunkSize not match drive chunkSize.")
                return ""
            chunk_size = max(available_chunk_size)
        else:
            chunk_size = max(self.chunk_size())
        drive_letter = self.info.get("drive_letter", "")
        sn = self.info.get("serial_number", "")
        target = self.info.get("target", "")
        ret_status, log_path = self.mcs.disk_log(drive_letter=drive_letter, target=target, sn=sn, chunk_size=chunk_size)
        return ret_status, log_path

    def chunk_size(self):
        return self.info.get("chunk_size")

    def upgrade(self, params):
        pass

    def farm_log(self):
        """
        Returns:
        """
        if self.info.get("vendor") == "SEAGATE":
            return self.mcs.get_seagate_farm_log(params=self.info)
        return ()

    def drive_letter(self):
        return self.info.get("drive_letter")

    def target(self):
        return self.info.get("target")

    def model_number(self):
        return self.info.get("model_number")

    def serial_number(self):
        return self.info.get("serial_number")

    def firmware_version(self):
        return self.info.get("firmware_version")

    def update_info(self):
        params = {
            "drive_letter": self.drive_letter(),
            "through_opt": self.target()
        }
        self.info.update(self.mcs.get_single_drive_info(params)[0])


class AtaDrive(Drive):
    """
    ATA Entity
    """

    def __init__(self, info):
        super(AtaDrive, self).__init__(info)

    def smart(self):
        command = ["info show-dev",
                   "info show-rotation-rate",
                   "info show-ata-identify",
                   "info show-ata-smart",
                   "attr smart-status -m show",
                   "log smart-health",
                   "log show-ata-error",
                   "log show-ata-extend-error",
                   "log show-ata-extend-dst",
                   "log show-ata-phy-event",
                   ]
        self.mcs.get_disk_smart(self.drive_letter(), self.target(), self.serial_number(), command)

    def upgrade(self, params):
        """ATA disk firmware upgrade"""
        upgrade_params = {
            "drive_letter": self.info.get("drive_letter"),
            "through_opt": self.target()
        }
        upgrade_params.update(params)
        if "pmc" in self.target():
            upgrade_params.update({"fragment": "32768"})
        self.mcs.upgrade_sata_drive(upgrade_params)


class ScsiDrive(Drive):
    """
    SCSI Entity
    """

    def __init__(self, info):
        super(ScsiDrive, self).__init__(info)

    def smart(self):
        command = ["info show-dev",
                   "info show-rotation-rate",
                   "attr show-disk-erc",
                   "attr smart-status -m show",
                   "log smart-health",
                   "log show-scsi-supported-log",
                   "log show-scsi-error-count",
                   "log show-scsi-bms",
                   "log show-scsi-dst",
                   "log show-scsi-ie",
                   "log show-scsi-temperature",
                   "info show-defect-list -t grown",
                   "info show-defect-list -t primary"
                   ]
        self.mcs.get_disk_smart(self.drive_letter(), self.target(), self.serial_number(), command)

    def upgrade(self, params):
        """SCSI disk firmware upgrade"""
        upgrade_params = {
            "drive_letter": self.info.get("drive_letter"),
            "through_opt": self.target()
        }
        upgrade_params.update(params)
        # 穿透PMC卡升级固件不支持64K分片，随机选择常用的32K
        if "pmc" in self.target():
            upgrade_params.update({"fragment": "32768"})
        self.mcs.upgrade_sas_drive(upgrade_params)


class NvmeDrive(Drive):
    """
    NVME Entity
    """

    def __init__(self, info):
        super(NvmeDrive, self).__init__(info)

    def smart(self):
        log_ids = self.mcs.get_pcie_support_log(self.drive_letter())
        command = ["info show-nvme-identify -t controller",
                   "info show-nvme-identify -t namespace",
                   "log show-nvme-log-supported"]
        for log_id in log_ids:
            command.append("log show-nvme-log -p {}".format(log_id))
        self.mcs.get_disk_smart(self.drive_letter(), self.target(), self.serial_number(), command)

    def upgrade(self, params):
        """NVMe disk firmware upgrade"""
        base_params = {
            "drive_letter": self.info.get("drive_letter")
        }
        base_params.update(params)
        self.mcs.download_nvme_firmware(base_params)
        self.mcs.active_nvme_firmware(base_params)
