#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 提供直通操作基础wrapper

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2022/09/12 18:00 created

"""
import codecs
import os
import platform
import re
import shutil
import stat
import time

from lib.exception.dfs_exception import DFSException
from lib.utils.command import CommandUtil
from lib.utils.log import SingleLog
from lib.utils.validate import ValidateUtil


class McsCli(object):

    def __init__(self):
        mcs_path = {
            "aarch64": os.path.join(".", "tools", "arm_64", "mcscli"),
            "x86_64": os.path.join(".", "tools", "x86_64", "mcscli")
        }
        self.log = SingleLog()
        self.tool_path = mcs_path.get(platform.machine(), "")
        os.chmod(self.tool_path, 755)
        os.chmod(self.tool_path, stat.S_IRWXU | stat.S_IRWXO)
        self.smart_log_path = os.path.join(".", "smart_log")

    @staticmethod
    def _parser_ret(raw_output):
        result = []
        for line in raw_output.split("\n"):
            ret = re.search(
                r"(/dev/\S+)\s+(--|[a-z]+:\d+(:\d+)?)\s+([A-Z]+)\s+([A-Ze]+)\s+([A-Za-z_]+)\s+(\S+\s*?\S+)"
                r"\s+(\S+)\s+(\S+)\s+(\d+)\s+([0-9-]+)\s+(\d+)\s+([A-Z-]+)\s+(\d+)\s+(\d+)", line)
            if ret:
                parser_ret = {
                    "drive_letter": ret.group(1),
                    "target": ret.group(2),
                    "medium_type": ret.group(4),
                    "protocol": ret.group(5).lower(),
                    "vendor": ret.group(6),
                    "model_number": ret.group(7),
                    "serial_number": ret.group(8),
                    "firmware_version": ret.group(9),
                    "lsector": ret.group(10),
                    "psector": ret.group(11),
                    "meta": ret.group(12),
                    "pilocation": ret.group(13),
                    "piType": ret.group(14),
                    "sectorcount": ret.group(15),
                    "capacity": str(int(ret.group(10)) * int(ret.group(15)))
                }
                result.append(parser_ret)
        return result

    def get_drives_info(self):
        """
        Get the info of all drive.
        """
        cmd = "{} info show-all-dev".format(self.tool_path)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            return []
        return self._parser_ret(res[1])

    def get_single_drive_info(self, params):
        """
        Get the info of the single drive.
        """
        drive_letter = params.get("drive_letter")
        through_opt = params.get("through_opt")
        cmd = "{} info show-dev".format(self.tool_path)
        if through_opt != "--":
            cmd += " -d {}".format(through_opt)
        cmd += " {}".format(drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        return self._parser_ret(res[1])

    @ValidateUtil.validate_param(params=dict)
    def upgrade_sas_drive(self, params):
        """
        SCSI disk firmware upgrade
        """
        ret = {"status_code": "0", "output": {}}
        drive_letter = params.get("drive_letter")
        file_path = params.get("file_path")
        through_opt = params.get("through_opt")
        # value range 0x04(NotFragment),0x05(NotFragment),0x06,0x07,0x0e(activation),0x0d(activation)
        upgrade_mode = params.get("upgrade_mode", "0x07")
        # multiple of 512
        fragment = params.get("fragment", "65536")
        # mode out of range
        upgrade_mode_out_range = upgrade_mode not in ("0x04", "0x05", "0x06", "0x07", "0x0d", "0x0e")
        if not all([file_path, drive_letter, through_opt]):
            raise DFSException(
                "The parameter is incorrect, , drive letter and firmware file and through option must be given!")
        if upgrade_mode_out_range:
            raise DFSException(
                "The parameter is incorrect, The upgrade mode must be within (0x04, 0x05, 0x06, 0x07, 0x0d, 0x0e)!")

        cmd = "{} fw update-scsi-fw -m {} -f {}".format(self.tool_path, upgrade_mode, file_path)
        if upgrade_mode in ("0x04", "0x05"):
            cmd += " -c 0"
        else:
            cmd += " -c {}".format(fragment)
        if through_opt != "--":
            cmd += " -d {}".format(through_opt)
        cmd += " {}".format(drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        return ret

    @ValidateUtil.validate_param(params=dict)
    def active_sas_firmware(self, params):
        """
        Activating the SCSI disk firmware
        """
        ret = {"status_code": "0", "output": {}}
        drive_letter = params.get("drive_letter")
        if not drive_letter:
            raise DFSException("The parameter is incorrect, please check.")
        cmd = "{} fw active-scsi-fw {}".format(self.tool_path, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        return ret

    @ValidateUtil.validate_param(params=dict)
    def upgrade_sata_drive(self, params):
        """
        ATA disk firmware upgrade
        """
        ret = {"status_code": "0", "output": {}}
        drive_letter = params.get("drive_letter")
        file_path = params.get("file_path")
        through_opt = params.get("through_opt")
        # value range 0x03,0x07(NotFragment),0x0e(activation)
        upgrade_mode = params.get("upgrade_mode", "0x03")
        # multiple of 512
        fragment = params.get("fragment", "65536")
        # PIO(0) or DMA(1)
        command_mode = params.get("command_mode", "0")

        if not all((file_path, drive_letter, through_opt)):
            raise DFSException(
                "The parameter is incorrect, drive letter and firmware file and through option must be given!")
        if upgrade_mode not in ("0x03", "0x07", "0x0e"):
            raise DFSException(
                "The parameter is incorrect, The upgrade mode must be within (0x03, 0x07, 0x0e)!")

        cmd = "{} fw update-ata-fw -m {} -f {} -t {}".format(
            self.tool_path, upgrade_mode, file_path, command_mode)
        if upgrade_mode == "0x07":
            cmd += " -c 0"
        else:
            cmd += " -c {}".format(fragment)
        if through_opt != "--":
            cmd += " -d {}".format(through_opt)
        cmd += " {}".format(drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        return ret

    @ValidateUtil.validate_param(params=dict)
    def active_sata_firmware(self, params):
        """
        Activating the ATA disk firmware
        """
        ret = {"status_code": "0", "output": {}}
        drive_letter = params.get("drive_letter")
        if not drive_letter:
            raise DFSException("The parameter is incorrect, please check.")
        cmd = "{} fw active-ata-fw -t 1 {}".format(self.tool_path, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        return ret

    @ValidateUtil.validate_param(params=dict)
    def download_nvme_firmware(self, params):
        """
        NVMe disk firmware download
        """
        ret = {"status_code": "0", "output": {}}
        file_path = params.get("file_path")
        drive_letter = params.get("drive_letter")
        if not file_path or not drive_letter:
            raise DFSException("The parameter is incorrect, please check.")
        cmd = "{} fw download-nvme-fw -c 65536 -f {} {}".format(self.tool_path, file_path, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        return ret

    @ValidateUtil.validate_param(params=dict)
    def active_nvme_firmware(self, params):
        """
        Activating the NVMe disk firmware
        """
        ret = {"status_code": "0", "output": {}}
        slot_number = params.get("slot_number", "0")
        active_mode = params.get("active_mode", "3")
        drive_letter = params.get("drive_letter")
        # mode out of range
        active_mode_out_range = active_mode not in ("0", "1", "2", "3")
        # slot out of range
        slot_number_out_range = slot_number not in ("0", "1", "2", "3", "4", "5", "6", "7")
        if not drive_letter or active_mode_out_range or slot_number_out_range:
            raise DFSException("The parameter is incorrect, please check.")
        cmd = "{} fw active-nvme-fw -m {} -s {} {}".format(self.tool_path, active_mode, slot_number, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        return ret

    @ValidateUtil.validate_param(params=dict)
    def firmware_slot_info(self, params):
        """
        Get firmware slot info
        """
        ret = {"status_code": "0", "output": {"slot_number": []}}
        drive_letter = params.get("drive_letter")
        if not drive_letter:
            raise DFSException("The parameter is incorrect, please check.")
        cmd = "{} log show-nvme-log -p 0x03 {}".format(self.tool_path, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        if not res[0]:
            raise DFSException(res[1])
        ret.get("output")["slot_number"] = re.findall(r"\n(\d)\s+", res[1])
        return ret

    def disk_log(self, drive_letter, target, sn, chunk_size=65536):
        '''
        Args:
            drive_letter: 盘符
            target: raid卡
            sn: 盘序列号
            chunk_size: 收集日志的块大小
        Returns:
            命令执行状态结果，日志路径
        '''
        log_path = os.path.join("/", "opt", "mcs", "collect_log", sn)
        if target != '--':
            cmd = "{} log get-disk-log -c {} -d {} {}".format(self.tool_path, chunk_size, target, drive_letter)
        else:
            cmd = "{} log get-disk-log -c {} {}".format(self.tool_path, chunk_size, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        return res[0], log_path

    def get_disk_smart(self, drive_letter, target, serial_number, command):
        # 每一个盘对应一个smart_log下的文件夹以SN命名
        # 创建文件,将文件目录存放在smart_log/sn/sn_time
        drive_smart_log_path = os.path.join(self.smart_log_path, serial_number)
        if os.path.exists(drive_smart_log_path):
            shutil.rmtree(drive_smart_log_path)
        os.mkdir(drive_smart_log_path)
        drive_smart_log_file = os.path.join(drive_smart_log_path,
                                            "{}_{}".format(serial_number,
                                                           time.strftime("%Y_%m_%d_%H_%M_%S", time.localtime())))
        with codecs.open("{}.txt".format(drive_smart_log_file), mode='a', encoding='utf-8') as f:
            for cmd in command:
                if target != '--':
                    cmd = "{} {} -d {} {}".format(self.tool_path, cmd, target, drive_letter)
                else:
                    cmd = "{} {} {}".format(self.tool_path, cmd, drive_letter)
                res, out, err = CommandUtil.exec_shell(cmd)
                if not res:
                    self.log.error("{} execute error, the command may not support by disk {}.".format(cmd,
                                                                                                      serial_number))
                    f.write("{} execute error, stdout info: {} error info : {}\n".format(cmd, out, err))
                f.write(u"{}\n".format(out))

    def is_support_internal_log(self, target, drive_letter):
        """
            Is collecting disk internal log supported
        Returns:
            boolean
        """
        if target != '--':
            cmd = "{} log show-disk-log-supported -d {} {}".format(self.tool_path, target, drive_letter)
        else:
            cmd = "{} log show-disk-log-supported {}".format(self.tool_path, drive_letter)
        res, stdout, _ = CommandUtil.exec_shell(cmd)
        if re.search(r"Supported disk vendor type", stdout):
            return True
        return False

    def get_seagate_farm_log(self, params):
        target = params.get("target", "")
        sn = params.get("serial_number", "")
        drive_letter = params.get("drive_letter", "")
        if target != '--':
            cmd = "{} log get-disk-log -t farm -d {} {}".format(self.tool_path, target, drive_letter)
        else:
            cmd = "{} log get-disk-log -t farm {}".format(self.tool_path, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        # 可能只收集cur，而factory收集失败
        if not res[0]:
            self.log.error("{} execute error, the command may not support by disk {}.".format(cmd, sn))
            return ()
        sn_dir = os.path.join("/", "opt", "mcs", "collect_log", sn)
        cur_farm = os.path.join(sn_dir, "farm.bin")
        factory_farm = os.path.join(sn_dir, "farm_factory.bin")
        previous_farm = os.path.join(sn_dir, "farm_previous.bin")
        return cur_farm, factory_farm, previous_farm

    def get_pcie_support_log(self, drive_letter):
        cmd = "{} log show-nvme-log-supported {}".format(self.tool_path, drive_letter)
        res = CommandUtil.exec_shell(cmd)
        # 命令执行失败
        if not res[0]:
            self.log.error("{} execute error, the command may not support by disk.".format(cmd))
            return []
        log_id = []
        for line in res[1].split("\n"):
            ret = re.search(r"(\S+)\s+support\s+support\s+\S+", line)
            if ret:
                log_id.append(ret.group(1))
        return log_id
