#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: 服务入口

版权信息: 华为技术有限公司，版权所有(C) 2022-2025

修改记录: 2023/7/11 18:00 created

"""
import json
import os
import platform
import re
import shutil
import sys
import time
import xml.etree.ElementTree as et

from config.constants import ReturnStatus, SPECIAL_DISK_UPGRADE_PARAMS
from lib.core.drives.drive import DriveFactoryMap
from lib.core.tools.mcscli import McsCli
from lib.core.tools.pack import Tar
from lib.core.tools.pack import Zip
from lib.exception.dfs_exception import DFSException
from lib.utils.csv_util import CsvUtil
from lib.utils.log import SingleLog
from lib.utils.progress import ProgressManager

G_PY_VERSION = platform.python_version().split(".")[0]


class Service(object):
    def __init__(self, filter_sn, filter_model):
        self.fil_sn = filter_sn
        self.fil_model = filter_model
        self.mcs = McsCli()
        self.drives = []
        self.project_path = os.path.realpath(os.path.dirname(os.path.dirname(__file__)))
        self.log = SingleLog()
        self.internal_progress_file = os.path.join(".", "internal_progress.txt")
        self.upgrade_progress_file = os.path.join("upgrade_process.txt")
        self.tar = Tar()
        self.zip = Zip()
        self.return_status = {}
        self.dfs_log = os.path.join(".", "dfs.log")
        self.upgrade_log = os.path.join(".", "internal_progress.txt")
        self.mcs_log_path = os.path.join("/", "opt", "mcs", "log", "mcslog.txt")
        self.sys_message_path = os.path.join("/", "var", "log", "messages")

    def get_all_disk(self):
        self.log.info("start get_all_disk")
        self._target_disk()
        try:
            fw_config_info_list, scene = self._get_all_upgrade_config()
        except DFSException:
            fw_config_info_list = []
            scene = ""
        info = []
        for drive in self.drives:
            model_number = drive.model_number()
            firmware_version = drive.firmware_version()
            sn = drive.serial_number()
            match_info_list = self._match_upgrade_config(
                model_number, firmware_version, fw_config_info_list, scene)
            firmware_target_version = "notFound"
            if match_info_list:
                firmware_target_version = match_info_list[-1].get("TargetVersion")
            active = "false"
            match_model = SPECIAL_DISK_UPGRADE_PARAMS.get(model_number)
            if match_model and match_model.get("active") and firmware_version in match_model.get("currentFW"):
                active = "true"
            info.append("{}:{}:{}:{}:{}".format(sn, model_number, firmware_version, firmware_target_version, active))
        if info:
            result = {"status": ReturnStatus.SUCCESS, "info": "\n".join(info)}
        else:
            result = {"status": ReturnStatus.FAIL, "desc": "No target disk"}
        self._output(result)

    def smart(self):
        self.log.info("start collect smart log")
        self._target_disk()
        # 默认日志存放路径 ./smart_log/ 即当前路径下的smart_log
        log_path = self.mcs.smart_log_path
        if os.path.exists(log_path):
            shutil.rmtree(log_path)
        os.mkdir(log_path)
        for drive in self.drives:
            # 收集硬盘SMART
            drive.smart()
            # 收集硬盘FARM LOG - 此日志收集时间较短,故放在SMART信息收集项完成
            farm_log_path = drive.farm_log()
            for f_log in farm_log_path:
                if os.path.exists(f_log):
                    # 移动farm_log文件到对应的smart目录 smart_log/sn/
                    shutil.move(f_log, os.path.join(log_path, drive.serial_number()))
        self.log.info("mcs collect smart log done.")
        self._save_log_file(log_path)
        self.log.info("start to make_archive.")
        shutil.make_archive(base_name="smart_log", format='tar', root_dir=log_path)
        cur_file = os.path.abspath(os.path.join(".", "smart_log.tar"))
        result = {"status": ReturnStatus.SUCCESS, "file_path": cur_file}
        self._output(result)
        self.log.info("complete collect smart service.")

    def internal_log(self):
        try:
            self.log.info("start collect internal log")
            self._target_disk()
            if not self.drives:
                result = {"status": ReturnStatus.FAIL, "desc": "No target disk"}
                self._output(result)
                return
            collect_time = int(time.time())
            pack_internal_log = "internal_log_{}".format(collect_time)
            os.mkdir(pack_internal_log)
            pack_logs = os.path.join(".", pack_internal_log)
            pm = ProgressManager(self.internal_progress_file)
            sn_list = [drive.serial_number() for drive in self.drives]
            pm.add_batch(sn_list)
            cal_timeout = len(sn_list) * 300
        except Exception as err:
            result = {"status": ReturnStatus.FAIL, "desc": str(err)}
            self._output(result)
            return
        result = {"status": ReturnStatus.SUCCESS, "timeout": cal_timeout}
        self._output(result)
        for drive in self.drives:
            sn = drive.serial_number()
            drive_letter = drive.drive_letter()
            target = drive.target()
            if not self.mcs.is_support_internal_log(target, drive_letter):
                # mcs不支持升级结果设置为成功
                self.log.warning(
                    "sn:{},model:{},not support collect internal log".format(sn, drive.model_number()))
                pm.set_success(sn)
                continue
            res, log_path = drive.internal()
            if os.path.exists(log_path):
                shutil.move(log_path, pack_logs)
            if res:
                pm.set_success(sn)
            else:
                pm.set_fail(sn, "collect internal Fail,see dfs.log for details")
        self._save_log_file(pack_logs)
        pack_all_name = "internal_log"
        shutil.make_archive(base_name=pack_all_name, format='tar', root_dir=pack_internal_log)
        shutil.rmtree(pack_logs)
        self.log.info("complete collect internal log")

    def internal_progress(self):
        """
        Returns:
            {"status":"success","file_path":"/root/path/to/internal_log.tar"}
            {"status":"failure","desc":"xxxxx","file_path":"/root/path/to/internal_log.tar"}
            {"status":"executing","progress":"43"}
        """
        pm = ProgressManager(self.internal_progress_file)
        proc_result = pm.get_process()
        file_path = os.path.join(".", "internal_log.tar")
        abs_file_path = os.path.abspath(file_path)
        status = proc_result.get("status")
        info = proc_result.get("info")
        ret = {}
        if status == ReturnStatus.SUCCESS:
            ret = {"status": ReturnStatus.SUCCESS, "file_path": abs_file_path}
        elif status == ReturnStatus.FAIL:
            ret = {"status": ReturnStatus.FAIL, "file_path": abs_file_path, "desc": info}
        elif status == ReturnStatus.RUNNING:
            ret = {"status": ReturnStatus.RUNNING, "progress": info}
        self._output(ret)

    def upgrade_firmware(self, fw_path=""):
        """hard disk upgrade firmware"""
        self.log.info("start upgrade firmware")
        self._target_disk()
        try:
            fw_config_info_list, scene = self._get_all_upgrade_config(fw_path)
        except DFSException as error:
            return_info = {
                "status": ReturnStatus.FAIL,
                "desc": error.message
            }
            self._output(return_info)
            return 1
        if len(self.drives) == 0:
            return_info = {
                "status": ReturnStatus.FAIL,
                "desc": "Hard disk was not found!"
            }
            self._output(return_info)
            return 1
        self._create_process(self.upgrade_progress_file)
        # spend time, cal 30s per disk
        spend_time = 30 * len(self.drives)
        return_info = {
            "status": ReturnStatus.SUCCESS,
            "timeout": spend_time
        }
        self._output(return_info)
        find_error = False
        record = False
        for drive in self.drives:
            model_number = drive.model_number()
            firmware_version = drive.firmware_version()
            sn = drive.serial_number()
            if find_error:
                msg = "Failed to upgrade the firmware of some disks, stop the upgrade."
                if not record:
                    self.log.error(msg)
                    record = True
                self.pm.set_fail(sn, msg)
                continue
            match_info_list = self._match_upgrade_config(
                model_number, firmware_version, fw_config_info_list, scene)
            if not match_info_list:
                # three situations: model is not configured
                # or firmware is not configured
                # or firmware is the target version
                msg = "[Model:%s][Firmware:%s]not found in config or is target version, skip upgrade!" % \
                      (model_number, firmware_version)
                self.log.warning(msg)
                self.pm.set_success(sn)
                continue
            # upgrade
            find_error = self._exec_upgrade(match_info_list, drive, find_error)
        self.log.info("upgrade complete")

    def _save_log_file(self, dest_dir):
        '''
        dfs.log internal_progress.txt mcslog.txt messages save to log dir
        Args:
            dest_dir:
        '''
        if os.path.exists(self.dfs_log):
            shutil.copy(self.dfs_log, dest_dir)
        if os.path.exists(self.upgrade_log):
            shutil.copy(self.upgrade_log, dest_dir)
        if os.path.exists(self.mcs_log_path):
            shutil.copy(self.mcs_log_path, dest_dir)
        if os.path.exists(self.sys_message_path):
            shutil.copy(self.sys_message_path, dest_dir)

    def _exec_upgrade(self, match_info_list, drive, find_error=False):
        model_number = drive.model_number()
        sn = drive.serial_number()
        for match_info in match_info_list:
            params = {
                "file_path": match_info.get("FirmwareFile")
            }
            if model_number in SPECIAL_DISK_UPGRADE_PARAMS:
                params.update(SPECIAL_DISK_UPGRADE_PARAMS[model_number])
            before_firmware = drive.firmware_version()
            try:
                drive.upgrade(params)
                drive.update_info()
            except DFSException as error:
                self.log.warning("[Model:%s]Failed to execute the firmware upgrade command, %s-->%s, msg:%s"
                                 % (model_number, before_firmware, match_info.get("TargetVersion"), error.message))
            after_firmware = drive.firmware_version()
            # check firmware
            if after_firmware == match_info.get("TargetVersion"):
                self.log.info("[Model:%s]upgrade successfully, %s-->%s" %
                              (model_number, before_firmware, match_info.get("TargetVersion")))
                self.pm.set_success(sn)
            else:
                if params.get("active") == "PowerOff" and before_firmware in params.get("currentFW"):
                    msg = "[Model:%s]upgrade successfully, %s-->%s, " \
                          "but need power on and off for the firmware to take effect" % \
                          (model_number, before_firmware, match_info.get("TargetVersion"))
                    self.log.warning(msg)
                    self.pm.set_success(sn)
                    continue
                msg = "[Model:%s]upgrade failed, %s-->%s" % \
                      (model_number, before_firmware, match_info.get("TargetVersion"))
                self.log.error(msg)
                self.pm.set_fail(sn, msg)
                find_error = True
                break
        return find_error

    def _target_disk(self):
        # 下发扫盘命令,获取当前环境上所有的硬盘列表
        drives = self.mcs.get_drives_info()
        self.log.info(drives)
        # 如果用户输入的SN不为空,则使用SN进行过滤
        if self.fil_sn:
            drives = filter(lambda item: item.get("serial_number") in self.fil_sn, drives)
        # 如果用户输入的MN不为空,则使用MN进行过滤
        if self.fil_model:
            drives = filter(lambda item: item.get("model_number") in self.fil_model, drives)
        # 过滤之后的硬盘列表为最终的需要操作的列表
        for drive in drives:
            # 最终的信息组装在service的实例化中
            self.drives.append(DriveFactoryMap().get(drive))

    @staticmethod
    def _output(dict_msg):
        print(json.dumps(dict_msg))
        sys.stdout.flush()

    @staticmethod
    def _match_upgrade_config(model_number, firmware_version, all_config_info, scene):
        """match the upgrade configuration based on the disk information"""

        def _match_info(match_firmware):
            for item in all_config_info:
                condition_model = model_number == item.get("Model")
                condition_fw = match_firmware in item.get("VersionList")
                check_new_fw = match_firmware != item.get("TargetVersion")
                if scene == "storage" and condition_model and check_new_fw:
                    return item
                elif condition_model and condition_fw and check_new_fw:
                    return item
            return {}

        match_info_list = list()
        tmp_firmware = firmware_version
        while True:
            match_info = _match_info(tmp_firmware)
            if match_info and match_info not in match_info_list:
                match_info_list.append(match_info)
                # update firmware, lookup again.
                tmp_firmware = match_info.get("TargetVersion")
                continue
            else:
                break
        return match_info_list

    def _get_all_upgrade_config(self, fw_path=""):
        """get the firmware upgrade configuration from the software package"""
        if not fw_path:
            fw_path = os.path.join(self.project_path, "upgrade")
        if not os.path.exists(fw_path):
            raise DFSException("The config path is not found '%s'" % fw_path)
        # format of the computing package
        computing_match_style = re.compile(r"^Medium_\d+\.\d+\.\d+_Software_Linux_.*\.zip")
        # format of the storage package
        storage_match_style = re.compile(r"^diskfw_signature.zip|^hssd_diskfw_signature.zip")
        is_computing = False
        is_storage = False
        computing_package_list = list()
        storage_package_list = list()
        for file_name in os.listdir(fw_path):
            if computing_match_style.match(file_name):
                is_computing = True
                computing_package_list.append(file_name)
            elif storage_match_style.match(file_name):
                is_storage = True
                storage_package_list.append(file_name)
        if (is_computing and is_storage) or (len(storage_package_list) > 1):
            raise DFSException("The config in the path is not unique '%s'" % fw_path)

        fw_config_info_list = list()
        scene = ""
        if is_computing:
            scene = "computing"
            for file_name in computing_package_list:
                src_path = os.path.join(fw_path, file_name)
                dst_path = os.path.join(fw_path, "tmp")
                # decompress
                self.zip.dc_specify_dir_path(src_path, dst_path)
                org_config = self._read_xml(os.path.join(dst_path, "version.xml"))
                # assemble the full path of the firmware
                org_config.update({
                    "FirmwareFile": os.path.join(fw_path, "tmp", "fw", org_config["FirmwareFileName"])
                })
                fw_config_info_list.append(org_config)
        if is_storage:
            scene = "storage"
            for file_name in storage_package_list:
                # decompress zip
                src_path = os.path.join(fw_path, file_name)
                dst_path = os.path.join(fw_path, "tmp")
                self.zip.dc_specify_dir_path(src_path, dst_path)

                # decompress tgz
                src_path = os.path.join(dst_path, file_name.replace("zip", "tgz"))
                self.tar.dc_specify_dir_path(src_path, dst_path)

                # read config
                storage_config = self._parse_storage_config(os.path.join(dst_path, "hddfw.conf"))
                fw_config_info_list.extend(storage_config)

        if not (is_computing or is_storage):
            scene = "self"
            config_file = os.path.join(fw_path, "firmware_config.csv")
            if not os.path.isfile(config_file):
                raise DFSException("The config file is not found '%s'" % config_file)
            try:
                fw_config_info_list = CsvUtil.read_csv(config_file)
                for item in fw_config_info_list:
                    item.update({"FirmwareFile": os.path.join(fw_path, item["FirmwareFileName"])})
            except UnicodeDecodeError:
                raise DFSException("The config file format is incorrect!")

        return fw_config_info_list, scene

    @staticmethod
    def _read_xml(config_path):
        """parse version.xml"""
        ret = {
            "FirmwareFileName": "",
            "TargetVersion": "",
            "Model": "",
            "VersionList": "",
        }
        if not os.path.isfile(config_path):
            raise DFSException("Config file is not found! '%s'" % config_path)
        tree = et.ElementTree(file=config_path)
        for node in tree.iter():
            if node.tag == "FileName":
                ret["FirmwareFileName"] = node.text
            elif node.tag == "Version":
                ret["TargetVersion"] = node.text
            elif node.tag == "SupportModel":
                ret["Model"] = node.text
            elif node.tag == "RuleOldVersion":
                ret["VersionList"] = node.text.replace("|", ";")

        return ret

    @staticmethod
    def _parse_storage_config(config_path):
        """parse hddfw.conf"""
        output = list()
        if not os.path.isfile(config_path):
            raise DFSException("Config file is not found! '%s'" % config_path)
        if G_PY_VERSION == "3":
            with open(config_path, 'r', encoding="gbk") as conf:
                all_lines = conf.readlines()
        else:
            with open(config_path, 'r') as conf:
                all_lines = conf.readlines()
        for line in all_lines:
            if line.startswith("//"):
                continue
            split_line = line.split(":")
            output.append({
                "FirmwareFileName": split_line[2],
                "TargetVersion": split_line[1],
                "Model": split_line[0],
                "VersionList": "",
                "FirmwareFile": os.path.join(os.path.dirname(config_path), split_line[2])
            })
        return output

    def _create_process(self, progress_file):
        """create process"""
        self.pm = ProgressManager(progress_file)
        sn_list = [drive.serial_number() for drive in self.drives]
        self.pm.add_batch(sn_list)

    def get_upgrade_process(self):
        """
        get progress of firmware upgrade
        Returns:
            {"status":"failure", "desc":"sn exec upgrade cmd fail"}
        """
        if not os.path.exists(self.upgrade_progress_file):
            raise DFSException("Progress file is not found! '%s'" % self.upgrade_progress_file)
        pm = ProgressManager(self.upgrade_progress_file)
        proc_result = pm.get_process()
        status = proc_result.get("status")
        info = proc_result.get("info")
        ret = {}
        if status == ReturnStatus.SUCCESS:
            ret = {"status": ReturnStatus.SUCCESS}
        elif status == ReturnStatus.FAIL:
            ret = {"status": ReturnStatus.FAIL, "desc": info}
        elif status == ReturnStatus.RUNNING:
            ret = {"status": ReturnStatus.RUNNING, "progress": info}
        self._output(ret)
