#!/usr/bin/env python
# -*- coding: UTF-8 -*-

"""
功    能: logger工具类

版权信息: 华为技术有限公司，版权所有(C) 2023-2025

修改记录: 2023/8/10 9:00 created

"""

import fcntl

from config.constants import ReturnStatus

SUCCEED = ReturnStatus.SUCCESS
FAILED = ReturnStatus.FAIL
EXECUTING = ReturnStatus.RUNNING


class ProgressManager(object):
    """
        以sn的维度管理对硬盘操作的进度
    """

    def __init__(self, file_path):
        self.file_path = file_path

    def add_batch(self, sn_list):
        """
        Args:
            sn_list:需要跟踪sn的记录列表
        Returns:
            boolean
        """
        progress = {}
        for sn in sn_list:
            progress[sn] = [sn, EXECUTING, ""]
        self._flush_to_file(progress)

    def set_success(self, sn):
        """
        Args:
            sn:硬盘sn号
        Returns:
            boolean
        """
        cur_progress = {}
        with open(self.file_path, "r") as up_file:
            for line in up_file.readlines():
                cur_sn, cur_stat, cur_reason = line.split(";")
                cur_progress[cur_sn] = [cur_sn, cur_stat, cur_reason.strip()]
        cur_progress[sn] = [sn, SUCCEED, ""]
        self._flush_to_file(cur_progress)

    def set_fail(self, sn, reason):
        """
        Args:
            sn: 硬盘sn号
            reason: 失败原因
        Returns:
            boolean

        """
        # 进度文件通过分号来区分列数，原因中包含需要替换
        reason = reason.replace(";", ":")
        cur_progress = {}
        with open(self.file_path, "r") as up_file:
            for line in up_file.readlines():
                cur_sn, cur_stat, cur_reason = line.split(";")
                cur_progress[cur_sn] = [cur_sn, cur_stat, cur_reason.strip()]
        cur_progress[sn] = [sn, FAILED, reason]
        self._flush_to_file(cur_progress)

    def get_process(self):
        """
        Returns:
            {"status":"failed","info":"sn exec upgrade cmd fail"}
        """
        with open(self.file_path, "r") as upgrade_file:
            fcntl.flock(upgrade_file.fileno(), fcntl.LOCK_EX)
            lines = upgrade_file.readlines()
            content = "".join(lines)
        if EXECUTING in content:
            ug_cnt = float(content.count(EXECUTING))
            total = float(content.count("\n"))
            percent = 100 - int((ug_cnt / total) * 100)
            return {"status": EXECUTING, "info": str(percent)}
        if FAILED in content:
            all_reason = ""
            for line in lines:
                status = line.split(";")[1]
                if status == FAILED:
                    all_reason += line
            return {"status": FAILED, "info": all_reason}
        return {"status": SUCCEED, "info": ""}

    def _flush_to_file(self, progress):
        """
        Args:
            progress: {sn1:[sn1,stat1,reason1],sn2:[sn2,stat2,reason2]}
        Returns:
            None
        """
        with open(self.file_path, 'w') as up_file:
            fcntl.flock(up_file.fileno(), fcntl.LOCK_EX)
            for _, s_p in progress.items():
                up_file.write("{};{};{}\n".format(s_p[0], s_p[1], s_p[2]))
